<?php
session_start();

require '../db_connection.php';
$tagsFile = 'game_tags.txt';

if (!isset($_SESSION['developer_id'])) {
    header("Location: index.php");
    exit();
}

$developer_id = $_SESSION['developer_id'];
$error_message = '';
$success_message = '';
$mainUrl = "https://" . $_SERVER['HTTP_HOST'];

$conn = getDBConnection();

try {
    $stmt = $conn->prepare("SELECT * FROM developers WHERE id = :id");
    $stmt->bindParam(':id', $developer_id, PDO::PARAM_INT);
    $stmt->execute();
    $developer = $stmt->fetch(PDO::FETCH_ASSOC);

    $base_url = 'https://' . $_SERVER['HTTP_HOST'] . '/pc_launcher/php/uploads/game_avatars/';

    if (!$developer) {
        $error_message = "Developer not found.";
        session_destroy();
        header("Location: index.php");
        exit();
    }

    $tagsList = [];
    if (file_exists($tagsFile)) {
        $tagsContent = file_get_contents($tagsFile);
        $tagsList = explode(',', $tagsContent);
    }

    $maintenanceStmt = $conn->prepare("SELECT * FROM maintenance ORDER BY id DESC LIMIT 1");
    $maintenanceStmt->execute();
    $maintenance = $maintenanceStmt->fetch(PDO::FETCH_ASSOC);

    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['is_active'])) {
        $is_active = isset($_POST['is_active']) ? 1 : 0;
        $end_time = $is_active ? $_POST['end_time'] : NULL;

        if ($maintenance) {
            $setMaintenanceStmt = $conn->prepare("
                UPDATE maintenance 
                SET is_active = :is_active, end_time = :end_time 
                WHERE id = :id
            ");
            $setMaintenanceStmt->bindParam(':id', $maintenance['id'], PDO::PARAM_INT);
        } else {
            $setMaintenanceStmt = $conn->prepare("
                INSERT INTO maintenance (is_active, end_time) 
                VALUES (:is_active, :end_time)
            ");
        }

        $setMaintenanceStmt->bindParam(':is_active', $is_active, PDO::PARAM_INT);
        $setMaintenanceStmt->bindParam(':end_time', $end_time);

        if ($setMaintenanceStmt->execute()) {
            $success_message = "Maintenance mode updated successfully.";
            $maintenance['is_active'] = $is_active;
            $maintenance['end_time'] = $end_time;
        } else {
            $error_message = "Error updating maintenance mode in the database.";
        }
    }

    // Change developer username
    if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['change_username'])) {
        $new_username = $_POST['new_username'];
        $check_username_stmt = $conn->prepare("SELECT COUNT(*) FROM developers WHERE username = :username");
        $check_username_stmt->bindParam(':username', $new_username, PDO::PARAM_STR);
        $check_username_stmt->execute();
        $username_exists = $check_username_stmt->fetchColumn();

        if ($username_exists) {
            $error_message = "Username already exists. Please choose a different username.";
        } else {
            $update_username_stmt = $conn->prepare("UPDATE developers SET username = :username WHERE id = :id");
            $update_username_stmt->bindParam(':username', $new_username, PDO::PARAM_STR);
            $update_username_stmt->bindParam(':id', $developer_id, PDO::PARAM_INT);

            if ($update_username_stmt->execute()) {
                $success_message = "Username changed successfully.";
            } else {
                $error_message = "Error updating username in the database.";
            }
        }
    }

    // Handle avatar upload
    if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_FILES['avatar']) && $_FILES['avatar']['error'] === UPLOAD_ERR_OK) {
        $upload_dir = '../uploads/game_avatars/';
        
        if (!is_dir($upload_dir)) {
            if (!mkdir($upload_dir, 0755, true)) {
                $error_message = "Error creating folder for avatars.";
            }
        }

        $avatar_name = basename($_FILES['avatar']['name']);
        $avatar_path = $upload_dir . $avatar_name;
        
        if (move_uploaded_file($_FILES['avatar']['tmp_name'], $avatar_path)) {
            $update_avatar_stmt = $conn->prepare("UPDATE developers SET avatar = :avatar WHERE id = :id");
            $update_avatar_stmt->bindParam(':avatar', $avatar_path, PDO::PARAM_STR);
            $update_avatar_stmt->bindParam(':id', $developer_id, PDO::PARAM_INT);
            
            if ($update_avatar_stmt->execute()) {
                $success_message = "Avatar uploaded successfully.";
                $developer['avatar'] = $avatar_path;
            } else {
                $error_message = "Error updating avatar in the database.";
            }
        } else {
            $error_message = "Error uploading avatar.";
        }
    }

    // Get the current game of the day
    $currentGameOfTheDayQuery = $conn->prepare("
        SELECT 
            games.id, 
            games.title, 
            games.description, 
            games.game_avatar, 
            game_of_the_day.end_date 
        FROM 
            game_of_the_day 
        INNER JOIN 
            games ON game_of_the_day.game_id = games.id 
        WHERE 
            game_of_the_day.end_date > NOW() 
        ORDER BY 
            game_of_the_day.end_date DESC
        LIMIT 1
    ");
    $currentGameOfTheDayQuery->execute();
    $currentGame = $currentGameOfTheDayQuery->fetch(PDO::FETCH_ASSOC);

    // Processing the form for choosing the game of the day
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_game_day'])) {
        $game_id = (int)$_POST['game_id'];
        $end_date = $_POST['end_date'];

        $deleteExistingGameDayStmt = $conn->prepare("DELETE FROM game_of_the_day WHERE end_date > NOW()");
        $deleteExistingGameDayStmt->execute();

        $checkGameQuery = $conn->prepare("
            SELECT COUNT(*) 
            FROM game_of_the_day 
            WHERE game_id = :game_id AND end_date = :end_date
        ");
        $checkGameQuery->execute([':game_id' => $game_id, ':end_date' => $end_date]);

        if ($checkGameQuery->fetchColumn() == 0) {
            $insertGameDayStmt = $conn->prepare("
                INSERT INTO game_of_the_day (game_id, end_date) 
                VALUES (:game_id, :end_date)
            ");
            if ($insertGameDayStmt->execute([':game_id' => $game_id, ':end_date' => $end_date])) {
                $success_message = "Game of the Day added successfully.";
            } else {
                $error_message = "Error adding Game of the Day to the database: " . implode(", ", $insertGameDayStmt->errorInfo());
            }
        } else {
            $error_message = "This game is already set for the selected date.";
        }

        header("Location: " . $_SERVER['PHP_SELF']);
        exit;
    }

    // Update game
    if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['update_game_submit'])) {
        $game_id = $_POST['game_id'];
        $game_title = $_POST['update_game_title'];
        $game_description = $_POST['update_game_description'];
        $game_version = $_POST['game_version'];
        $is_editor_choice = isset($_POST['is_editor_choice']) ? 1 : 0;
        $game_avatar = '';
        $game_banner = '';
        $screenshots = [];

        // Processing Avatar Upload
        if (isset($_FILES['update_game_avatar']) && $_FILES['update_game_avatar']['error'] === UPLOAD_ERR_OK) {
            $avatar_name = basename($_FILES['update_game_avatar']['name']);
            $game_avatar = '../uploads/game_avatars/' . $avatar_name;
            move_uploaded_file($_FILES['update_game_avatar']['tmp_name'], $game_avatar);
        }

        // Processing banner upload
        if (isset($_FILES['update_game_banner']) && $_FILES['update_game_banner']['error'] === UPLOAD_ERR_OK) {
            $banner_name = basename($_FILES['update_game_banner']['name']);
            $game_banner = '../uploads/game_banners/' . $banner_name;
            move_uploaded_file($_FILES['update_game_banner']['tmp_name'], $game_banner);
        }

        // Processing screenshots
        if (isset($_FILES['update_game_screenshots'])) {
            $upload_dir = '../uploads/game_screenshots/';
            $total_files = count($_FILES['update_game_screenshots']['name']);
            for ($i = 0; $i < $total_files; $i++) {
                if ($_FILES['update_game_screenshots']['error'][$i] === UPLOAD_ERR_OK) {
                    $screenshot_name = basename($_FILES['update_game_screenshots']['name'][$i]);
                    $screenshot_file = $upload_dir . $screenshot_name;
                    if (move_uploaded_file($_FILES['update_game_screenshots']['tmp_name'][$i], $screenshot_file)) {
                        $screenshots[] = $screenshot_file;
                    }
                }
            }
        }

        // Updating the game in the database
        $update_stmt = $conn->prepare("UPDATE games SET title = :title, description = :description, version = :version, isEditorChoice = :is_editor_choice WHERE id = :game_id");
        $update_stmt->bindParam(':title', $game_title);
        $update_stmt->bindParam(':description', $game_description);
        $update_stmt->bindParam(':version', $game_version);
        $update_stmt->bindParam(':is_editor_choice', $is_editor_choice);
        $update_stmt->bindParam(':game_id', $game_id);
        $update_stmt->execute();

        if ($game_avatar) {
            $update_avatar_stmt = $conn->prepare("UPDATE games SET game_avatar = :game_avatar WHERE id = :game_id");
            $update_avatar_stmt->bindParam(':game_avatar', $game_avatar);
            $update_avatar_stmt->bindParam(':game_id', $game_id);
            $update_avatar_stmt->execute();
        }

        if ($game_banner) {
            $update_banner_stmt = $conn->prepare("UPDATE games SET game_banner = :game_banner WHERE id = :game_id");
            $update_banner_stmt->bindParam(':game_banner', $game_banner);
            $update_banner_stmt->bindParam(':game_id', $game_id);
            $update_banner_stmt->execute();
        }

        foreach ($screenshots as $screenshot) {
            $insert_screenshot_stmt = $conn->prepare("INSERT INTO game_screenshots (game_id, screenshot_url) VALUES (:game_id, :screenshot_url)");
            $insert_screenshot_stmt->bindParam(':game_id', $game_id);
            $insert_screenshot_stmt->bindParam(':screenshot_url', $screenshot);
            $insert_screenshot_stmt->execute();
        }

        $success_message = "Game updated successfully.";
    }

    // Delete game
    if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['delete_game_id'])) {
        $delete_game_id = $_POST['delete_game_id'];
        
        $delete_screenshots_stmt = $conn->prepare("DELETE FROM game_screenshots WHERE game_id = :game_id");
        $delete_screenshots_stmt->bindParam(':game_id', $delete_game_id, PDO::PARAM_INT);
        $delete_screenshots_stmt->execute();
        
        $delete_stmt = $conn->prepare("DELETE FROM games WHERE id = :game_id");
        $delete_stmt->bindParam(':game_id', $delete_game_id, PDO::PARAM_INT);
        
        if ($delete_stmt->execute()) {
            $success_message = "Game deleted successfully.";
        } else {
            $error_message = "Error deleting game from the database.";
        }
    }

    // Fetch developer's games
    $games = [];
    $stmt = $conn->prepare("SELECT * FROM games WHERE developer_id = :developer_id");
    $stmt->bindParam(':developer_id', $developer_id, PDO::PARAM_INT);
    $stmt->execute();
    $games = $stmt->fetchAll(PDO::FETCH_ASSOC);

} catch (PDOException $e) {
    $error_message = 'Database error: ' . $e->getMessage();
}
?>

<!DOCTYPE html>
<html lang="ru">
<head>
<meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Developer Console - PC Launcher</title>
    <link rel="shortcut icon" href="https://aisencorporation.com/icons/ais_corp.png" type="image/x-icon">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@400;600&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/dropzone/5.10.1/dropzone.min.css">
    <link rel="stylesheet" href="styles/dashboard.css">
    <script src="https://cdn.jsdelivr.net/npm/resumablejs@1.1.0/resumable.min.js"></script>
    <script src="https://ajax.googleapis.com/ajax/libs/jquery/3.6.0/jquery.min.js" defer></script>
    <script src="https://cdn.jsdelivr.net/npm/axios/dist/axios.min.js" defer></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/dropzone/5.10.1/dropzone.min.js" defer></script>
    <script src="https://unpkg.com/@yaireo/tagify" defer></script>
    <script src="main.js" defer></script>
</head>
<body>
    <div class="container">

        <div class="logout-block">
            <button class="logout" onclick="window.location.href = 'logout.php'">
                <div class="sign"><svg viewBox="0 0 512 512"><path d="M377.9 105.9L500.7 228.7c7.2 7.2 11.3 17.1 11.3 27.3s-4.1 20.1-11.3 27.3L377.9 406.1c-6.4 6.4-15 9.9-24 9.9c-18.7 0-33.9-15.2-33.9-33.9l0-62.1-128 0c-17.7 0-32-14.3-32-32l0-64c0-17.7 14.3-32 32-32l128 0 0-62.1c0-18.7 15.2-33.9 33.9-33.9c9 0 17.6 3.6 24 9.9zM160 96L96 96c-17.7 0-32 14.3-32 32l0 256c0 17.7 14.3 32 32 32l64 0c17.7 0 32 14.3 32 32s-14.3 32-32 32l-64 0c-53 0-96-43-96-96L0 128C0 75 43 32 96 32l64 0c17.7 0 32 14.3 32 32s-14.3 32-32 32z"></path></svg></div>
                
                <div class="text">Logout</div>
            </button>
        </div><br>

        <br><h1>Welcome, <?php echo htmlspecialchars($developer['username']); ?>!</h1>
        <p>Company: <?php echo htmlspecialchars($developer['companyname']); ?></p>
        <p>Developer ID: <?php echo $developer_id; ?></p>

        <?php if ($developer['avatar']): ?>
            <img src="<?php echo htmlspecialchars($developer['avatar']); ?>" alt="Avatar" class="avatar" width="150" height="150">
        <?php else: ?>
            <img src="default-avatar.png" alt="Avatar" class="avatar" width="150" height="150">
        <?php endif; ?>

        <?php if ($error_message): ?>
            <div class="message error">
                <span class="close-button">&times;</span>
                <?php echo $error_message; ?>
            </div>
        <?php endif; ?>
        <?php if ($success_message): ?>
            <div class="message success">
                <span class="close-button">&times;</span>
                <?php echo $success_message; ?>
            </div>
        <?php endif; ?>

        <div class="button-container">
            <button class="submit-button" onclick="toggleAvatarForm()">
                <span>Change developer avatar</span>
            </button>
            <div id="avatarForm" style="display: none;">
                <form id="avatarUploadForm" action="" method="post" enctype="multipart/form-data">
                    <input type="file" name="avatar" required>
                    <button type="submit" class="upload-game">Load developer avatar</button>
                </form>
            </div>

            <button class="submit-button" onclick="window.location.href = 'launcher_settings.php'">
                <span>Update Launcher</span>
            </button>

            <button class="submit-button" onclick="window.location.href = 'news.php'">
                <span>Manage launcher news</span>
            </button>

            <button class="submit-button" onclick="toggleUsernameInput()">
                <span>Change developer username</span>
            </button>

            <div id="username-change" style="display:none; margin-top: 10px;">
                <form action="" method="post" enctype="multipart/form-data">
                    <input type="text" id="new_username" placeholder="New Username" />
                    <button onclick="changeUsername()" class="upload-game">Confirm</button>
                </form>
            </div>

            <button class="submit-button" onclick="window.location.href = 'https://aisencorporation.com/contact'">
                <span>Technical support for developers</span>
            </button>
        </div><br>

        <div class="maintenance-control">
            <h2>Maintenance Mode</h2>

            <?php if ($maintenance && $maintenance['is_active']): ?>
                <p>Maintenance mode is currently <strong>enabled</strong>.</p>
                <p>End Time: <?php echo date('Y-m-d H:i:s', strtotime($maintenance['end_time'])); ?></p>
            <?php else: ?>
                <p>Maintenance mode is currently <strong>disabled</strong>.</p>
            <?php endif; ?>

            <form method="post" action="">
                <label>
                    <input type="checkbox" name="is_active" <?php if ($maintenance && $maintenance['is_active']) echo 'checked'; ?>>
                    Enable Maintenance Mode
                </label>
                <br>
                <label for="endTime">End Time:</label>
                <input type="datetime-local" name="end_time" value="<?php echo $maintenance ? date('Y-m-d\TH:i:s', strtotime($maintenance['end_time'])) : ''; ?>" step="1">
                <button type="submit" class="upload-game">Set Maintenance</button>
            </form>
        </div><br>

        <div class="load-new-game" onclick="toggleGameDay()">
            <h2>Game Day</h2>
            <div tabindex="0" class="plusButton">
                <svg class="plusIcon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 30 30">
                    <g mask="url(#mask0_21_345)">
                        <path d="M13.75 23.75V16.25H6.25V13.75H13.75V6.25H16.25V13.75H23.75V16.25H16.25V23.75H13.75Z"></path>
                    </g>
                </svg>
            </div>
        </div><br>

        <div id="gameDayBlock" style="display: none;">
            <?php if ($currentGame): ?>
                <div>
                    <h3>Current Game of the Day: <?= htmlspecialchars($currentGame['title']); ?></h3>
                    <div style="display: flex; justify-content: center;">
                        <img src="<?= htmlspecialchars($base_url . basename($currentGame['game_avatar'])); ?>" alt="Avatar" style="width:100px;height:auto;">
                    </div>
                    <p><?= htmlspecialchars($currentGame['description']); ?></p>
                    <p>End Date: <?= htmlspecialchars($currentGame['end_date']); ?></p>
                </div>
            <?php else: ?>
                <p>There is currently no game of the day.</p>
            <?php endif; ?>

            <form method="post" action="">
                <label for="game_id">Select Game:</label>
                <select id="game_id" name="game_id" required onchange="updateGameInfo()">
                    <option value="">Select a game</option>
                    <?php foreach ($games as $game): ?>
                        <option value="<?= htmlspecialchars($game['id']); ?>" 
                                data-avatar="<?= htmlspecialchars($base_url . basename($game['game_avatar'])); ?>"
                                data-description="<?= htmlspecialchars($game['description']); ?>">
                            <?= htmlspecialchars($game['title']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>

                <div id="game_info" style="margin-top: 10px; text-align: center;">
                    <img id="game_avatar" src="" alt="Game Avatar" style="width:100px;height:auto;display:none;">
                    <p id="game_description"></p>
                </div>

                <label for="event_date">Event Date:</label>
                <input type="date" id="event_date" name="end_date" required>

                <button type="submit" name="add_game_day" class="upload-game" id="submitBtn">Apply</button>
            </form><br>
        </div>
        
        <div class="load-new-game" onclick="LoadNewGame()">
            <h2>Load new game</h2>
            <div tabindex="0" class="plusButton">
                <svg class="plusIcon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 30 30">
                    <g mask="url(#mask0_21_345)">
                        <path d="M13.75 23.75V16.25H6.25V13.75H13.75V6.25H16.25V13.75H23.75V16.25H16.25V23.75H13.75Z"></path>
                    </g>
                </svg>
            </div>
        </div>

        <div id="uploadFormContainer" class="upload-form-container">        
            <form id="uploadForm" method="post" enctype="multipart/form-data" class="upload-form hidden">
                <label for="game_title">Game name:</label>
                <input type="text" id="game_title" name="game_title" placeholder="Enter the game name" required>
                
                <label for="game_description">Game description:</label>
                <textarea id="game_description" name="game_description" placeholder="Enter a description of the game" rows="5" required></textarea>
                
                <label for="game_version">Game Version:</label>
                <input type="text" id="game_version" name="game_version" placeholder="Enter game version" required>
                
                <label class="ui-bookmark" style="display: flex; align-items: center;">
                    Editor Choice:
                    <input type="checkbox" id="editor-choice" name="is_editor_choice" style="margin-left: 8px;">
                    <div class="bookmark" style="margin-left: 8px;">
                        <svg viewBox="0 0 32 32">
                            <g>
                                <path d="M27 4v27a1 1 0 0 1-1.625.781L16 24.281l-9.375 7.5A1 1 0 0 1 5 31V4a4 4 0 0 1 4-4h14a4 4 0 0 1 4 4z"></path>
                            </g>
                        </svg>
                    </div>
                </label><br>

                <label for="game_tags">Game tags (max 5):</label>
                <div id="tag-container">
                    <input type="hidden" id="game_tags_hidden" name="game_tags">
                    <input type="text" id="game_tags_input" placeholder="Введите тег" list="tagsList" required>
                    <datalist id="tagsList"></datalist>
                </div>
                <div id="selected-tags"></div>
                <p id="error-message" style="color: red;"></p>
                
                <label for="game_avatar" class="custom-avatar-label">Game Avatar:</label>
                <input type="file" id="game_avatar" name="game_avatar" accept="image/*" onchange="handleAvatarChange(event)" class="custom-avatar-input" required>
                <div id="avatar_preview"></div>

                <label for="game_banner">Game banner:</label>
                <label class="dropzone" id="bannerDropzone">
                    <span>Drag banner here or click to select</span>
                    <img id="bannerPreview" alt="Preview">
                    <input type="file" id="gameBannerInput" name="game_banner" accept="image/*" style="display:none" onchange="previewImage(event)" required>
                </label><br>

                <label for="orientation">Orientation:</label>
                <div class="radio-group">
                    <label class="radio-label">
                        <input type="radio" name="screenshot_orientation" value="horizontal" checked>
                        Horizontal
                        <span class="custom-radio"></span>
                    </label>
                    <label class="radio-label">
                        <input type="radio" name="screenshot_orientation" value="vertical">
                        Vertical
                        <span class="custom-radio"></span>
                    </label>
                </div>

                <label for="game_screenshots">Game screenshots:</label>
                <input type="file" id="game_screenshots" name="game_screenshots[]" accept="image/*" multiple required>
                <div id="screenshots_preview"></div>

                <div id="modal" class="modal" onclick="this.style.display='none';" style="display: none;">
                    <span class="close-modal" onclick="this.parentElement.style.display='none';">&times;</span>
                    <img id="modalImage" src="" alt="Modal Image" />
                </div>
                
                <button type="button" onclick="uploadMetadata()" class="upload-game">Upload Game Details</button>
            </form><br>
        </div>

        <form id="executableUploadForm" method="post" enctype="multipart/form-data" class="upload-form hidden">
            <label for="game_executable">Select Archive (.zip):</label>
            <input type="file" id="game_executable" name="game_executable" accept=".zip">
            <input type="hidden" id="game_executable_path" value="game_id_value_here">
            <button type="button" onclick="startUpload(document.getElementById('game_executable_path').value, 'game_executable')" class="upload-game">Upload Archive</button><br>

            <a href="#" id="toggleManualUpload">Upload the game archive manually (for huge files and more)</a>

            <div id="manualUpload" class="upload-section hidden">
                <ol>
                    <li>You will need an FTP client, such as the popular <a href="https://filezilla-project.org/" target="_blank" style="color: #007BFF; font-weight: bold;">FileZilla</a> or another client of your choice.</li>
                    <li>Open your FTP client and connect using your FTP credentials (username and password).</li>
                    <li>Once connected, navigate to the directory where the launcher is installed. The path is usually something like: <strong style="color: #FF6347;"><?php echo $mainUrl; ?>/pc_launcher/php/uploads/game_uploads/</strong>.</li>
                    <li>Drag and drop your <strong style="color: #FF6347;">.zip archive</strong> containing the game into this folder and wait for the upload to complete.</li>
                    <li>After the upload is complete, enter the exact name of the uploaded <strong style="color: #FF6347;">.zip archive</strong> in the input field below and click the "Submit Archive Name" button.</li>
                </ol><br>
                <label for="ftp_archive_name">Enter Archive Name (.zip):</label><br>
                <input type="text" id="ftp_archive_name" name="ftp_archive_name" placeholder="example.zip" style="width: 100%" required><br>
                <input type="hidden" id="game_id" value="game_id_value_here">
                <button type="button" onclick="submitToDatabase(document.getElementById('game_executable_path').value, document.getElementById('ftp_archive_name').value)" class="submit-ftp upload-game" style="width: 100%">Submit Archive Name</button><br>
                <p><strong style="color: #FF6347;">IMPORTANT!</strong> If you do not upload the archive but submit the form, it will result in a download error in the application.</p>
            </div>
        </form>

        <div id="progress-overlay" style="display: none;">
            <div id="progress-container">
                <div class="progress-circle">
                    <div class="progress-spinner" id="progress-spinner"></div>
                    <div class="progress-text" id="progress-text"></div>
                </div>
                <div class="progress-info">
                    <div id="speed" style="font-size: 18px;"></div>
                    <div id="remaining"></div>
                </div>
            </div>
        </div>

        <h2>Your games</h2>
        <table>
            <thead>
                <tr>
                    <th>ID</th>
                    <th>Game Name</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($games as $game): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($game['id']); ?></td>
                        <td><?php echo htmlspecialchars($game['title']); ?></td>
                        <td class="action-buttons">
                            <button class="edit-button" onclick="toggleUpdateForm(<?php echo $game['id']; ?>)">
                                <svg class="edit-svgIcon" viewBox="0 0 512 512">
                                    <path d="M410.3 231l11.3-11.3-33.9-33.9-62.1-62.1L291.7 89.8l-11.3 11.3-22.6 22.6L58.6 322.9c-10.4 10.4-18 23.3-22.2 37.4L1 480.7c-2.5 8.4-.2 17.5 6.1 23.7s15.3 8.5 23.7 6.1l120.3-35.4c14.1-4.2 27-11.8 37.4-22.2L387.7 253.7 410.3 231zM160 399.4l-9.1 22.7c-4 3.1-8.5 5.4-13.3 6.9L59.4 452l23-78.1c1.4-4.9 3.8-9.4 6.9-13.3l22.7-9.1v32c0 8.8 7.2 16 16 16h32zM362.7 18.7L348.3 33.2 325.7 55.8 314.3 67.1l33.9 33.9 62.1 62.1 33.9 33.9 11.3-11.3 22.6-22.6 14.5-14.5c25-25 25-65.5 0-90.5L453.3 18.7c-25-25-65.5-25-90.5 0zm-47.4 168l-144 144c-6.2 6.2-16.4 6.2-22.6 0s-6.2-16.4 0-22.6l144-144c6.2-6.2 16.4-6.2 22.6 0s6.2 16.4 0 22.6z"></path>
                                </svg>
                            </button>

                            <form action="" method="post" onsubmit="return confirmDelete();">
                                <input type="hidden" name="delete_game_id" value="<?php echo $game['id']; ?>">
                                <button type="submit" class="delete-button">
                                    <svg viewBox="0 0 448 512" class="svgIcon">
                                        <path d="M135.2 17.7L128 32H32C14.3 32 0 46.3 0 64S14.3 96 32 96H416c17.7 0 32-14.3 32-32s-14.3-32-32-32H320l-7.2-14.3C307.4 6.8 296.3 0 284.2 0H163.8c-12.1 0-23.2 6.8-28.6 17.7zM416 128H32L53.2 467c1.6 25.3 22.6 45 47.9 45H346.9c25.3 0 46.3-19.7 47.9-45L416 128z"></path>
                                    </svg>
                                </button>
                            </form>
                        </td>
                    </tr>
                    <tr id="updateForm<?php echo $game['id']; ?>" style="display: none;">
                        <td colspan="3">
                            <form action="" method="post" enctype="multipart/form-data" class="upload-form">
                                <input type="hidden" name="game_id" value="<?php echo $game['id']; ?>">
                                <label for="update_game_title">Game name:</label>
                                <input type="text" id="update_game_title" name="update_game_title" value="<?php echo htmlspecialchars($game['title']); ?>" required>

                                <label for="update_game_description">Game description:</label>
                                <textarea id="update_game_description" name="update_game_description" rows="5" required><?php echo htmlspecialchars($game['description']); ?></textarea>

                                <label for="update_game_version">Game Version:</label>
                                <input type="text" id="update_game_version" name="game_version" value="<?php echo htmlspecialchars($game['version']); ?>" required>

                                <label class="ui-bookmark" style="display: flex; align-items: center;">
                                    Editor Choice:
                                    <input type="checkbox" id="editor-choice" name="is_editor_choice" <?php echo $game['isEditorChoice'] ? 'checked' : ''; ?> style="margin-left: 8px;"/>
                                    <div class="bookmark" style="margin-left: 8px;">
                                        <svg viewBox="0 0 32 32">
                                            <g>
                                                <path d="M27 4v27a1 1 0 0 1-1.625.781L16 24.281l-9.375 7.5A1 1 0 0 1 5 31V4a4 4 0 0 1 4-4h14a4 4 0 0 1 4 4z"></path>
                                            </g>
                                        </svg>
                                    </div>
                                </label><br>

                                <label for="update_game_avatar">Game avatar:</label>
                                <input type="file" id="update_game_avatar" name="update_game_avatar" accept="image/*">

                                <label for="update_game_banner">Game banner:</label>
                                <input type="file" id="update_game_banner" name="update_game_banner" accept="image/*">

                                <label for="update_game_screenshots">Game screenshots:</label>
                                <input type="file" id="update_game_screenshots" name="update_game_screenshots[]" accept="image/*" multiple>

                                <button type="submit" name="update_game_submit" class="upload-game">Update game</button>
                            </form>

                            <form method="post" enctype="multipart/form-data" class="upload-form">
                                <label for="update_game_zip">Update ZIP File:</label>
                                <input type="file" id="update_game_zip_<?php echo $game['id']; ?>" name="update_game_zip" accept=".zip">
                                <button type="button" onclick="startUpload(<?php echo $game['id']; ?>, 'update_game_zip_<?php echo $game['id']; ?>')" class="upload-game">Upload Update</button>
                            </form>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
</body>
</html>